<?php
defined('ABSPATH') or die;

class NpAdminActions {

    private static $_editorPageTypes = array(
        '' => 'theme-template',
        'html' => 'np-template-header-footer-from-plugin',
        'html-header-footer' => 'np-template-header-footer-from-theme'
    );

    /**
     * Defines site is https and localhost
     *
     * @return bool
     */
    public static function siteIsSecureAndLocalhost() {
        return NpAdminActions::isSSL() && NpAdminActions::isLocalhost();
    }

    /**
     * Defines site is ssl
     *
     * @return bool
     */
    public static function isSSL()
    {
        $isSSL = false;

        if (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https') {
            $_SERVER['HTTPS'] = 'on';
        }

        if (isset($_SERVER['HTTPS'])) {
            if ('on' == strtolower($_SERVER['HTTPS'])) {
                $isSSL = true;
            }
            if ('1' == $_SERVER['HTTPS']) {
                $isSSL = true;
            }
        } elseif (isset($_SERVER['SERVER_PORT']) && ('443' == $_SERVER['SERVER_PORT'])) {
            $isSSL = true;
        }
        return $isSSL;
    }

    /**
     * Defines site is localhost
     *
     * @return bool
     */
    public static function isLocalhost()
    {
        $whitelist = array(
            // IPv4 address
            '127.0.0.1',
            // IPv6 address
            '::1'
        );

        if (filter_has_var(INPUT_SERVER, 'REMOTE_ADDR')) {
            $ip = filter_input(INPUT_SERVER, 'REMOTE_ADDR', FILTER_VALIDATE_IP);
        } else if (filter_has_var(INPUT_ENV, 'REMOTE_ADDR')) {
            $ip = filter_input(INPUT_ENV, 'REMOTE_ADDR', FILTER_VALIDATE_IP);
        } else if (isset($_SERVER['REMOTE_ADDR'])) {
            $ip = filter_var($_SERVER['REMOTE_ADDR'], FILTER_VALIDATE_IP);
        } else {
            $ip = null;
        }
        return $ip && in_array($ip, $whitelist);
    }

    /**
     * Filter on get_edit_post_link
     * Add domain parameter for debug
     *
     * @param string $link
     *
     * @return string
     */
    public static function editPostLinkFilter($link) {
        $domain = NpEditor::getDomain();
        if ($domain) {
            $link = add_query_arg(array('domain' => urlencode($domain)), $link);
        }
        return $link;
    }

    /**
     * Action on edit_form_top
     * Add domain hidden field to editor form
     */
    public static function editFormTopFilter() {
        $domain = NpEditor::getDomain();
        if ($domain) {
            printf('<input type="hidden" name="domain" value="%s" />', $domain);
        }
    }

    /**
     * Action on themler_edit_form_buttons
     * Add "Edit with Mediaspot-Zopyron" button
     *
     * @param WP_Post $post
     */
    public static function addMediaspot_zopyronButtonAction($post) {
        if (!NpEditor::isAllowedForEditor($post)) {
            return;
        }
        if (np_data_provider($post->ID)->isConvertRequired()) {
            ?>
            <a href="#" id="convert-in-mediaspot_zopyron" class="button mediaspot_zopyron-editor"><?php _e('Turn to Mediaspot-Zopyron', 'mediaspot_zopyron'); ?></a>
            <?php
        } else {
            ?>
            <a href="#" id="edit-in-mediaspot_zopyron" class="button mediaspot_zopyron-editor"><?php _e('Edit with Mediaspot-Zopyron', 'mediaspot_zopyron'); ?></a>
            <?php $autoSaveChanges = !!get_post_meta($post->ID, '_np_html_auto_save', true);
            if ($autoSaveChanges) { ?>
                <p><b><?php echo __('The page has unpublished changes.', 'mediaspot_zopyron');?></b> <?php echo __('Click \'Edit\' with Mediaspot_zopyron\' button and then click \'Publish\' to make them visible on your website.', 'mediaspot_zopyron');?></p>
            <?php }
        }
    }

    /**
     * Action on admin_menu
     * Add Mediaspot_zopyron menus and register page capabilities
     */
    public static function addEditorPageAction() {
        $capability = 'edit_pages';
        $menu_slug = 'np_app';
        add_pages_page(__('Mediaspot-Zopyron', 'mediaspot_zopyron'), __('Mediaspot-Zopyron', 'mediaspot_zopyron'), $capability, $menu_slug, 'np_start');
        add_pages_page(__('Mediaspot-Zopyron', 'mediaspot_zopyron'), __('Mediaspot-Zopyron', 'mediaspot_zopyron'), $capability, 'np_editor', 'NpAdminActions::editorAction');

        add_submenu_page($menu_slug, __('Import', 'mediaspot_zopyron'), __('Import', 'mediaspot_zopyron'), $capability, 'np_import', 'NpImport::importPage');
        add_submenu_page($menu_slug, __('Plugin Wizard', 'mediaspot_zopyron'), __('Plugin Wizard', 'mediaspot_zopyron'), $capability, 'np_wizard', 'NpImport::wizardPage');
        add_submenu_page($menu_slug, __('Settings', 'mediaspot_zopyron'), __('Settings', 'mediaspot_zopyron'), $capability, 'np_settings', 'NpSettings::settingsPage');

        // remove submenu from Pages
        global $submenu;
        $pages_slug = 'edit.php?post_type=page';
        if (isset($submenu[$pages_slug]) && is_array($submenu[$pages_slug])) {
            foreach ($submenu[$pages_slug] as $key => $value) {
                if (in_array('np_editor', $value) || in_array('np_app', $value)) {
                    unset($submenu[$pages_slug][$key]);
                }
            }
        }
    }

    /**
     * Action on custom_menu_order
     * Add Mediaspot_zopyron menus and register page capabilities
     *
     * @return bool
     */
    public static function addSubmenusAction() {
        global $submenu;
        $menu_slug = NpAdminActions::adminUrlFilter(admin_url('post-new.php?post_type=page&np_new=1'));
        $menu_slug_2 = 'Colors and Fonts';
        $capability = 'edit_pages';
        $submenu[$menu_slug][10] = array(__('Settings', 'mediaspot_zopyron'), $capability, admin_url('admin.php?page=np_settings'));
        $submenu[$menu_slug][11] = array(__('New Page', 'mediaspot_zopyron'), $capability, $menu_slug);
        $submenu[$menu_slug][12] = array(__('Import', 'mediaspot_zopyron'), $capability, admin_url('admin.php?page=np_import'));
        $submenu[$menu_slug_2][13] = array(__('Colors', 'mediaspot_zopyron'), $capability, $menu_slug . '&np_page=colors');
        $submenu[$menu_slug_2][14] = array(__('Fonts', 'mediaspot_zopyron'), $capability, $menu_slug . '&np_page=fonts');
        $submenu[$menu_slug_2][15] = array(__('Typography', 'mediaspot_zopyron'), $capability, $menu_slug . '&np_page=typography');
        $submenu[$menu_slug_2][16] = array(__('Headings and Text', 'mediaspot_zopyron'), $capability, $menu_slug . '&np_page=customize');
        $submenu[$menu_slug_2][17] = array(__('Header', 'mediaspot_zopyron'), $capability, $menu_slug . '&np_page=Header');
        $submenu[$menu_slug_2][18] = array(__('Menu Style', 'mediaspot_zopyron'), $capability, $menu_slug . '&np_page=Menu');
        $submenu[$menu_slug_2][19] = array(__('Footer', 'mediaspot_zopyron'), $capability, $menu_slug . '&np_page=Footer');
        $submenu[$menu_slug][20] = array(__('Plugin Wizard', 'mediaspot_zopyron'), $capability, admin_url('admin.php?page=np_wizard'));
        return true;
    }

    /**
     * Action on
     *  _network_admin_menu,
     *  _user_admin_menu,
     *  _admin_menu
     *
     * Add Mediaspot_zopyron menus and register page capabilities
     */
    public static function addMenuAction() {
        global $menu;
        $menu_slug = NpAdminActions::adminUrlFilter(admin_url('post-new.php?post_type=page&np_new=1'));
        $menu_slug_2 = 'Colors and Fonts';
        $capability = 'edit_pages';
        $menu['56'] = array(__('Mediaspot_zopyron', 'mediaspot_zopyron'), $capability, $menu_slug, '', 'menu-top menu-icon-mediaspot_zopyron', 'menu-mediaspot_zopyron', 'div');
        $menu['58'] = array(__('Colors and Fonts', 'mediaspot_zopyron'), $capability, $menu_slug_2, '', 'menu-top menu-icon-mediaspot_zopyron-2', 'menu-mediaspot_zopyron-2', 'div');
    }

    public static $thumbnailScale = 0.3;

    /**
     * Action on admin_print_styles
     */
    public static function printMenuStylesAction() {
        ?>
        <style>
            #adminmenu .menu-icon-mediaspot_zopyron div.wp-menu-image {
                background-image: url('<?php echo APP_PLUGIN_URL; ?>editor/assets/images/menu-icon.png');
                background-position:50% 55%;
                background-repeat: no-repeat;
                background-size: 16px;
            }

            #adminmenu .menu-icon-mediaspot_zopyron-2 div.wp-menu-image {
                background-image: url('<?php echo APP_PLUGIN_URL; ?>editor/assets/images/colors and fonts.png');
                background-position:50% 55%;
                background-repeat: no-repeat;
                background-size: 15px;
            }

            #adminmenu .menu-icon-mediaspot_zopyron:hover div.wp-menu-image,
            #adminmenu .menu-icon-mediaspot_zopyron.wp-has-current-submenu div.wp-menu-image,
            #adminmenu .menu-icon-mediaspot_zopyron.current div.wp-menu-image {
                opacity: 0.85;
            }

            .wp-core-ui .button.mediaspot_zopyron-editor {
                background: #4082f3;
                color: #fff;
                font-size: 14px;
                height: 46px;
                line-height: 44px;
                padding: 0 41px 0 36px;
            }

            .mediaspot_zopyron-editor:before {
                background-image: url('<?php echo APP_PLUGIN_URL; ?>editor/assets/images/edit-menu-icon.png');
                display: inline-block;
                margin-right: 5px;
                content: "";
                background-size: 19px;
                width: 19px;
                height: 19px;
                background-repeat: no-repeat;
                vertical-align: middle;
                margin-bottom: 3px;
            }

            .edit-post-header-toolbar>.components-button.mediaspot_zopyron-editor {
                display: inline-flex;
                background: #3f82f4;
                color: #fff;
                padding: 5px 13px 8px 9px;
                font-size: 14px;
            }

            .components-icon-button.mediaspot_zopyron-editor:not(:disabled):not([aria-disabled=true]):not(.is-default):hover,
            .components-icon-button.mediaspot_zopyron-editor:not(:disabled):not([aria-disabled=true]):not(.is-default):active,
            .components-icon-button.mediaspot_zopyron-editor:not(:disabled):not([aria-disabled=true]):not(.is-default):focus,
            .wp-core-ui .button.mediaspot_zopyron-editor:hover,
            .wp-core-ui .button.mediaspot_zopyron-editor:active,
            .wp-core-ui .button.mediaspot_zopyron-editor:focus {
                background: #2b7aff;
                box-shadow: none;
                color: #fafafa;
            }

            .components-button.mediaspot_zopyron-editor:before {
                margin-bottom: 0;
                width: 19px;
                height: 19px;
                background-size: 19px;
            }

            #mediaspot_zopyron-preview-frame {
                transform: scale(<?php echo self::$thumbnailScale; ?>);
                transform-origin: 0 0;
                height: <?php echo 100 / self::$thumbnailScale; ?>%;
            }

            #mediaspot_zopyron-preview {
                margin-left: auto;
                margin-right: auto;
                cursor: pointer;
                overflow: hidden;
                height: 0;
            }

            #mediaspot_zopyron-preview-frame {
                pointer-events: none;
            }
            #np-loader {
                position: absolute;
                visibility: hidden;
                width: 1800px;
            }
        </style>
        <?php
    }


    /**
     * Get Mediaspot_zopyron editor iframe html
     *
     * @param array $args
     *
     * @return string
     */
    public static function getEditorContainerHtml($args = array()) {
        ob_start();
        ?>
        <div id="mediaspot_zopyron-editor-container" style="display: block; z-index: 100; position: relative;">
            <style>
                html {
                    overflow: hidden !important;
                }
                #wpcontent {
                    padding-left: 0 !important;
                }
                #mediaspot_zopyron-editor-frame {
                    width: 100%;
                    height: calc(100vh - 32px); /*for admin bar*/
                }
            </style>
            <script>
                jQuery(document).scroll(function() {
                    if (jQuery('#mediaspot_zopyron-editor-container').length) {
                        jQuery(this).scrollTop(0);
                    }
                });
            </script>
            <iframe id="mediaspot_zopyron-editor-frame" src="<?php echo NpEditor::getAppLink($args); ?>"></iframe>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Action on load-appearance_page_np_editor
     * Print Mediaspot_zopyron editor iframe
     *
     * @deprecated
     */
    public static function editorAction() {
        throw new Exception("Deprecated usage");
    }

    /**
     * Print Mediaspot_zopyron editor page
     */
    public static function appAction() {
        include_once dirname(__FILE__) . '/mediaspot_zopyron-editor.php';
        die();
    }

    /**
     * Action on edit_form_after_title
     * Print page preview (first 4 sections)
     *
     * @param WP_Post $post
     */
    public static function showScreenShotsAction($post) {
        $data_provider = np_data_provider($post->ID);
        if ($data_provider->isMediaspot_zopyron()) {
            ob_start();
            ?>
            <head>
                <link rel="stylesheet" type="text/css" media="all" href="<?php echo APP_PLUGIN_URL; ?>assets/css/mediaspot_zopyron.css">

                <?php echo $data_provider->getPageFonts(); ?>
                <style>
                    <?php echo $data_provider->getStyleCss(); ?>
                    <?php echo $data_provider->getPageHead(); ?>
                </style>
            </head>
            <?php
            $frame_head = ob_get_clean();
            ob_start();
            ?>
            <body class="<?php echo $data_provider->getPageBodyClass(); ?>" style="<?php echo $data_provider->getPageBodyStyle(); ?>">
            <?php
            self::$_previewSectionIdx = 0;
            echo preg_replace_callback(
                '#<section [\s\S]*?</section>#',
                'NpAdminActions::_screenshotSectionsReplacer',
                $data_provider->getPagePublishHtml()
            );
            ?>
            </body>
            <?php
            $frame_body = ob_get_clean();
            ?>
            <div id="mediaspot_zopyron-preview">
                <iframe id="mediaspot_zopyron-preview-frame">
                </iframe>
            </div>
            <style>
                #postdivrich {
                    display: none;
                }
            </style>
            <script>
                jQuery(function ($) {
                    var doc = $('#mediaspot_zopyron-preview-frame')[0].contentDocument;
                    doc.open();
                    doc.write("<!DOCTYPE html>\n<html>\n" + <?php echo wp_json_encode($frame_head); ?> + <?php echo wp_json_encode($frame_body); ?> + "</html>");
                    doc.close();

                    $("#mediaspot_zopyron-preview-frame").load(function() {
                        var totalHeight = 0,
                            width = 1600;

                        $('#mediaspot_zopyron-preview-frame').css('width', width + 'px');
                        $('#mediaspot_zopyron-preview').css('width', width * <?php echo self::$thumbnailScale; ?> + 'px');
                        $('#mediaspot_zopyron-preview-frame').contents().find('section').each(function () {
                            totalHeight += $(this).height();
                        });
                        $('#mediaspot_zopyron-preview').css('height', totalHeight * <?php echo self::$thumbnailScale; ?>);
                    });
                });
            </script>
            <?php
        }
    }

    /**
     * @param array $m
     *
     * @return string
     */
    public static function _screenshotSectionsReplacer($m) {
        if (self::$_previewSectionIdx >= 4) {
            return '';
        }
        self::$_previewSectionIdx++;
        return $m[0];
    }

    private static $_previewSectionIdx;


    /**
     * Action on admin_head
     * Print page editor scripts and styles
     */
    public static function printPreviewStylesAction() {
        global $post;

        if ($post && $post->ID) {
            if (!NpEditor::isAllowedForEditor($post)) {
                return;
            }
            np_data_provider($post->ID, true)->clear();
        }
        ?>
        <style>
            #mediaspot_zopyron-preview {
                font-size: 0;
                line-height: 0;
            }
            #mediaspot_zopyron-preview a img {
                max-width: 100%;
            }
        </style>

        <script>
            var menuFolded;

            function collapseMenu() {
                jQuery('body').addClass('folded');
                jQuery(document).trigger('wp-collapse-menu', {state: 'folded'});
            }
            function startEditor(t) {
                function runMediaspot_zopyron() {
                    if (window.dataBridge) {
                        menuFolded = jQuery('body').hasClass('folded');
                        jQuery(window).scrollTop(0);
                        var editorContainer = <?php echo wp_json_encode(self::getEditorContainerHtml(array('post_id' => $post ? $post->ID : 0))); ?>;
                        jQuery('#wpbody-content').prepend(editorContainer);
                        collapseMenu();
                    } else {
                        alert('Unable to start the Editor. Please contact the Support.');
                    }
                }

                function redirectWhenSave()
                {
                    setTimeout(function() {
                            wp.data.select("core/editor").isSavingPost() ? redirectWhenSave() : runMediaspot_zopyron()}
                        , 300);
                }
                t.preventDefault();
                //for old wp version
                if(wp.data === undefined) {
                    /*var $wpTitle = jQuery('#title');
                    if (!$wpTitle.val()) {
                        $wpTitle.val('Page #' + jQuery('#post_ID').val());
                    }
                    if (wp.autosave) {
                        wp.autosave.server.triggerSave();
                    }
                    jQuery(document).on('heartbeat-tick.autosave', function () {
                        jQuery(window).unbind('beforeunload');
                        runMediaspot_zopyron();
                    });*/
                    runMediaspot_zopyron();
                } else { //for wp version 5.0 and more
                    var isNewPage = '<?php echo (isset($_SERVER['SCRIPT_FILENAME']) && strpos($_SERVER['SCRIPT_FILENAME'], 'post-new.php') !== false) ? '1' : '0'; ?>';
                    if(isNewPage === '1') {
                        wp.data.select("core/editor").getEditedPostAttribute("title") || wp.data.dispatch("core/editor").editPost({
                            title: "Page #" + jQuery("#post_ID").val()
                        });
                        wp.data.dispatch("core/editor").savePost();
                        redirectWhenSave();
                    } else {
                        runMediaspot_zopyron();
                    }
                }
            }
            function closeEditor(data) {
                if (data.needRefresh) {
                    if (!data.closeUrl || window.location.href === data.closeUrl) {
                        window.location.reload();
                    } else {
                        window.location.href = data.closeUrl;
                    }
                } else {
                    jQuery('#mediaspot_zopyron-editor-container').remove();
                    jQuery('body').toggleClass('folded', menuFolded);
                }
            }

            function createAnchorListItem(anchor) {
                if (!anchor) {
                    return anchor;
                }
                var result = jQuery('<li><input type="hidden" class="item-permalink"><span class="item-title"></span><span class="item-info">Section</span></li>');
                result.children('input').attr('value', anchor.url);
                result.children('span.item-title').text(anchor.title);
                return result;
            }

            function addListItem(table, element) {
                table.find('ul').append(element);
            }

            function fixAlternateStyles(table) {
                table.find('li:even').addClass('alternate');
                table.find('li:odd').removeClass('alternate');
            }

            function addAnchorLinks(table, data) {
                if (!data || !data.length) {
                    return;
                }
                var listItems = jQuery.map(data, createAnchorListItem);
                jQuery.each(listItems, function (i, item) {
                    addListItem(table, item);
                });
                fixAlternateStyles(table);
            }

            function selectListItemByUrl(list, url) {
                var matchUrl = function (index, element) {
                    return jQuery(element).children('.item-permalink').prop('value') === url;
                }
                var listItems = list.find('li'),
                    active = listItems.filter(matchUrl);
                if (active.hasClass('selected')) {
                    return;
                }
                listItems.removeClass('selected');
                active.addClass('selected');
            }

            function handleAnchorListItemClick(event) {
                var element = jQuery(this),
                    dialog = element.closest('#wp-link'),
                    list = element.closest('ul');
                wpLink.updateFields(event, element);
                selectListItemByUrl(list, jQuery('#wp-link-url').prop('value'));
            }

            function handleAnchorLinkCheckboxChange(event) {
                var input = jQuery(this),
                    dialog = input.closest('#wp-link'),
                    list = dialog.find('#anchor-links-list'),
                    value = input.attr('value'),
                    searchField = dialog.find('.link-search-wrapper label');
                if (value === "section") {
                    list.removeClass('hidden');
                    searchField.addClass('hidden');
                    selectListItemByUrl(list, jQuery('#wp-link-url').prop('value'));
                } else {
                    list.addClass('hidden');
                    searchField.removeClass('hidden');
                }
            }

            function markAnchorByUrl(dialog, url) {
                if (url && url.indexOf('#') === 0 && url !== '#') {
                    dialog.find('.anchor-link input[value=section]').click();
                } else {
                    dialog.find('.anchor-link input[value=page]').click();
                }
            }

            function addAnchorsToLinkDialog(dialog, data) {
                if (!data || !data.l || !data.l.anchorLink || !data.anchorsList || !data.anchorsList.length) {
                    return;
                }
                dialog.find('.link-target').after(
                    '<div class="anchor-link" style="margin-top: 5px"><label><span style="vertical-align: bottom">' +
                    data.l.Destination +
                    '</span>\n<input type="radio" name="link-destination" value="page"> ' +
                    data.l.pageLink +
                    '</label><br><label><span></span>\n<input type="radio" name="link-destination" value="section"> ' +
                    data.l.anchorLink +
                    '</label></div>'
                );
                dialog.find('#search-panel').append(
                    '<div id="anchor-links-list" class="query-results hidden" tabindex="0"><ul></ul>' +
                    '<div class="river-waiting"><span class="spinner"></span></div></div>'
                );
                dialog.find('#anchor-links-list').on('click', 'li', handleAnchorListItemClick);
                dialog.find('.anchor-link input').on('change', handleAnchorLinkCheckboxChange);
                addAnchorLinks(dialog.find('#anchor-links-list'), data.anchorsList);
                markAnchorByUrl(dialog, data.url);
            }

            function cleanupLinkDialog(dialog) {
                dialog.find('.anchor-link input').off('change', handleAnchorLinkCheckboxChange);
                dialog.find('#anchor-links-list').off('click', 'li', handleAnchorListItemClick);
                dialog.find('.anchor-link, #anchor-links-list').remove();
            }

            function postMessageListener(event) {
                if (event.origin !== location.origin) {
                    return;
                }
                var data;
                try {
                    data = JSON.parse(event.data);
                }
                catch(e){
                }
                if (!data) {
                    return;
                }
                var childWindow = jQuery('#mediaspot_zopyron-editor-frame')[0].contentWindow;

                if (data.action === 'close') {
                    closeEditor(data);
                } else if (data.action === 'editLinkDialogOpen') {
                    wpLink.open();
                    jQuery('.wp-link-text-field, #wplink-link-existing-content').remove();

                    jQuery('#wp-link-target').prop('checked', !!data.data.blank);
                    jQuery('#wp-link-url').prop('value', data.data.url);

                    addAnchorsToLinkDialog(jQuery('#wp-link'), data.data);

                    var wpLinkUpdate = wpLink.update;
wpLink.update = function() {
    var attrs = wpLink.getAttrs();
    var $hrefDom = jQuery('#wp-link').find('.selected [value="' + attrs.href + '"]').closest('.selected');
    var value = $hrefDom.find('.item-title').text();
    var data = {
        url: attrs.href,
        blank: !!attrs.target
    };
    if (value) {
        data.value = value;
    }
    childWindow.postMessage(JSON.stringify({
        action: 'editLinkDialogClose',
        data: data
    }), window.location.origin);
    wpLink.update = wpLinkUpdate;
    wpLink.close();
};

                    jQuery(document).one('wplink-close', function () {
                        cleanupLinkDialog(jQuery('#wp-link'));
                        childWindow.postMessage(JSON.stringify({action: 'editLinkDialogClose'}), window.location.origin);
                    });
                }
            }

            function waitFor(selector, callback) {
                var el = jQuery(selector);

                if (el.length) {
                    callback(el);
                    return;
                } else {
                    setTimeout(function() {
                        waitFor(selector, callback);
                    }, 500);
                }
            }

            jQuery(function ($) {
                menuFolded = $('body').hasClass('folded');

                var buttonHtml = '<button type="button" class="components-button components-icon-button mediaspot_zopyron-editor">' +
                    '<?php echo __('Turn to Mediaspot-Zopyron', 'mediaspot_zopyron') ?>' +
                    '</button>';

                <?php if (version_compare(get_bloginfo('version'), '5.0-beta5', '>=')) : ?>
                $(function () {
                    waitFor('.edit-post-header-toolbar', function (toolbar) {
                        toolbar
                            .append(buttonHtml)
                            .on('click', '.mediaspot_zopyron-editor', startEditor);
                    });
                });
                <?php endif ?>

                $('#edit-in-mediaspot_zopyron, #convert-in-mediaspot_zopyron, #mediaspot_zopyron-preview').click(startEditor);

                if (location.search.indexOf('np_new=') !== -1 || location.search.indexOf('np_edit=') !== -1) {
                    collapseMenu();
                }

                <?php if (_arr($_GET, 'np_new')) : ?>
                $('.wp-menu-open').removeClass('wp-has-current-submenu wp-menu-open').addClass('wp-not-current-submenu');
                $('#menu-mediaspot_zopyron, #menu-mediaspot_zopyron > a').removeClass('wp-not-current-submenu').addClass('wp-has-current-submenu wp-menu-open');
                <?php endif; ?>
            });

            if (window.addEventListener) {
                window.addEventListener("message", postMessageListener);
            } else {
                window.attachEvent("onmessage", postMessageListener); // IE8
            }

        </script>
        <?php
    }

    /**
     * Get menu items from wp to editor
     *
     * @return array
     */
    public static function getMenuItems() {

        $result = array();
        $locations = get_nav_menu_locations();
        for ($i = 0; $i < count($locations); $i++) {
            $menu_id = array_pop($locations);
            if ($menu_id) {
                break;
            }
        }
        if (isset($menu_id)) {
            $items = wp_get_nav_menu_items($menu_id);
            if ($items) {
                $result = self::buildHierarchicalMenu($items);
            }
        }
        return count($result) > 0 ? $result : null;
    }

    /**
     * Build menu items in hierarchical array
     *
     * @param array $items
     * @param int   $parentId
     *
     * @return array
     */
    public static function buildHierarchicalMenu(array $items, $parentId = 0) {
        $branch = array();

        foreach ($items as $item) {
            $link_title = $item->post_title ? $item->post_title : $item->title;
            $item_id = (get_the_ID() === (int) $item->object_id) ? (int) $item->object_id : false;
            $result = array(
                'title' => $link_title,
                'id' => $item_id
            );

            if ($item->menu_item_parent == $parentId) {
                $children = self::buildHierarchicalMenu($items, $item->ID);
                if ($children) {
                    $result['items'] = $children;
                }
                $branch[] = $result;
            }
        }
        return $branch;
    }

    /**
     * Action on admin_head
     * Print prepared data for Mediaspot_zopyron-editor
     *
     * @throws Exception
     */
    public static function printBridgeAction() {
        global $post;
        if (!$post || !NpEditor::isAllowedForEditor($post)) {
            return;
        }
        $user = wp_get_current_user();
        $uid = (int)$user->ID;
        $ajax_nonce = wp_create_nonce('np-upload');
        $edit_post_url = NpAdminActions::adminUrlFilter(admin_url('post.php?action=edit')) . '&post={id}';
        $autoSave = get_option('np_include_auto_save') === '1' ? true : false;

        $settings = array(
            'actions' => array(
                'uploadImage'            => NpAction::getActionUrl('np_upload_image'),
                'savePage'               => NpAction::getActionUrl('np_save_page'),
                'clearChunks'            => NpAction::getActionUrl('np_clear_chunks'),
                'saveSiteSettings'       => NpAction::getActionUrl('np_save_site_settings'),
                'getSite'                => NpAction::getActionUrl('np_get_site'),
                'getSitePosts'           => NpAction::getActionUrl('np_get_site_posts'),
                'savePreferences'        => NpAction::getActionUrl('np_save_preferences'),
                'routeServiceWorker'     => NpAction::getActionUrl('np_route_service_worker'),
            ),
            'ajaxData' => array(
                'uid' => $uid,
                '_ajax_nonce' => $ajax_nonce,
            ),
            'uploadImageOptions' => array(
                'formFileName' => 'async-upload',
                'params' => array(
                    'html-upload' => 'Upload',
                    '_wpnonce' => wp_create_nonce('media-form'),
                    'uid' => $uid,
                    '_ajax_nonce' => $ajax_nonce,
                ),
            ),
            'editPostUrl' => $edit_post_url,
            'dashboardUrl' => NpAdminActions::adminUrlFilter(admin_url()),
            'loginUrl' => wp_login_url($edit_post_url),
            'pageId' => np_data_provider($post->ID)->isConvertRequired() ? '' : $post->ID,
            'startPageId' => $post->ID,
            'startPageTitle' => $post->post_title,
        );

        $cms_settings = array(
            'defaultImageUrl' => NpAttachments::getDefaultImageUrl(),
            'defaultLogoUrl' => NpAttachments::getDefaultLogoUrl(),
            'isFirstStart' => !!_arr($_GET, 'np_new'),
            'maxRequestSize' => min(
                wp_convert_hr_to_bytes(ini_get('post_max_size')),
                wp_convert_hr_to_bytes(ini_get('upload_max_filesize')),
                wp_convert_hr_to_bytes(ini_get('memory_limit'))
            ),
            'isWhiteLabelPlugin' => pathinfo(dirname(dirname(__FILE__)), PATHINFO_BASENAME) != ('n' . 'i' . 'c' . 'e' . 'p' . 'a' . 'g' . 'e'),
            'disableAutosave' => !$autoSave
        );
        $data_provider_auto_save = np_data_provider($post->ID, null, false);
        $data_provider = np_data_provider($post->ID);

        $page_obj = NpAction::getPost($post);
        $page_html_auto_save = $data_provider_auto_save->getPageHtml();
        $page_html = $data_provider->getPageHtml();
        $page_html = ($page_html_auto_save && $page_html_auto_save !== '') ? $page_html_auto_save : $page_html;
        $page_html = self::_restorePageType($page_html, $post);
        $data = array(
            'site' => NpGetSiteAction::getSite(),
            'pageHtml' => $page_html,
            'nicePageCss' => file_get_contents(APP_PLUGIN_PATH . 'assets/css/mediaspot_zopyron-dynamic.css'),
            'downloadedFonts' => self::getDownloadedFonts(),
        );
        $page_found = false;
        foreach ($data['site']['items'] as $page) {
            if ($page['id'] === $page_obj['id']) {
                $page_found = true;
                break;
            }
        }
        $headerNp_auto_save = get_option('headerNp_auto_save');
        $headerNp = get_option('headerNp');
        $headerNp = ($headerNp_auto_save && $headerNp_auto_save !== '') ? $headerNp_auto_save : $headerNp;
        $footerNp_auto_save = get_option('footerNp_auto_save');
        $footerNp = get_option('footerNp');
        $footerNp = ($footerNp_auto_save && $footerNp_auto_save !== '') ? $footerNp_auto_save : $footerNp;
        if (isset($headerNp)) {
            $headerItem = json_decode($headerNp, true);
            $data['site']['header'] = $headerItem['html'];
        }
        if (isset($footerNp)) {
            $footerItem = json_decode($footerNp, true);
            $data['site']['footer'] = $footerItem['html'];
        }

        if (!$page_found) {
            array_push($data['site']['items'], $page_obj);
        }

        $newPageTitle = '';
        if (!$page_obj['title']) {
            $uniqueTitle = self::_createUniqueTitle('Page');
            $page_obj['title'] = $uniqueTitle;
            $newPageTitle = $uniqueTitle;
        }

        $data['info'] = array(
            'taxonomies' => array(),
            'menus' => array(),
            'productsExists' => class_exists('WooCommerce'),
            'newPageUrl' => NpAdminActions::adminUrlFilter(admin_url('post-new.php?post_type=page&np_new=1')),
            'forceModified' => !get_post_meta($post->ID, '_np_site_style_css_used_ids', true),
            'generalSettingsUrl' => NpAdminActions::adminUrlFilter(admin_url('options-general.php')),
            'typographyPageHtmlUrl' => add_query_arg(array('np_html' => '1', 'preview' => 'true'), get_permalink($post->ID)),
            'siteIsSecureAndLocalhost' => NpAdminActions::siteIsSecureAndLocalhost(),
            'newPageTitle' => $newPageTitle,
            'fontsInfo' => self::getFontsInfo(),
            'menuItems' => self::getMenuItems(),
        );
        foreach (get_taxonomies(array('show_tagcloud' => true), 'object') as $taxonomy) {
            $data['info']['taxonomies'][] = array('name' => $taxonomy->name, 'label' => $taxonomy->label);
        }
        foreach (wp_get_nav_menus() as $menu) {
            $data['info']['menus'][] = array('id' => $menu->term_id . '', 'label' => $menu->name);
        }
        //if true editor can start
        if ($settings && $cms_settings && $data) {
            update_option('npDataBridge', 1);
        } else {
            update_option('npDataBridge', 0);
        }
        ?>
        <script>
            var dataBridgeData = <?php echo wp_json_encode($data, JSON_PRETTY_PRINT); ?>,
                callbacks = [],
                attemptsCount = 0;

            window.dataBridge = {
                getSite: function getSite() {
                    return dataBridgeData.site;
                },
                setSite: function setSite(site) {
                    dataBridgeData.site = site;
                },
                getPageHtml: function getPageHtml() {
                    return dataBridgeData.pageHtml;
                },
                getNPCss: function getNPCss() {
                    return dataBridgeData.nicePageCss;
                },
                getDownloadedFonts: function getDownloadedFonts() {
                    return dataBridgeData.downloadedFonts;
                },
                setDownloadedFonts: function setDownloadedFonts(downloadedFonts) {
                    dataBridgeData.downloadedFonts = downloadedFonts;
                },
                getStartTerm: function getStartTerm() {
                    return "<?php echo $data_provider->isConvertRequired() ? 'site:wordpress:' . $post->ID : ''; ?>";
                },
                getDefaultPageType: function getDefaultPageType() {
                    return "<?php echo self::$_editorPageTypes[NpSettings::getOption('np_template')]; ?>";
                },
                getInfo: function getInfo() {
                    return dataBridgeData.info;
                },
                doLoggedIn: function doLoggedIn(func) {
                    callbacks.push(func);

                    // show login dialog
                    if (jQuery('#wp-auth-check-wrap').hasClass('hidden')) {
                        if (attemptsCount > 0) {
                            // login dialog will not be shown a second time
                            // see  "$(document).off( 'heartbeat-tick.wp-auth-check' );" in wp-auth-check.js
                            callbacks.forEach(function (cb) {
                                cb(false);
                            });
                            callbacks = [];
                            return;
                        }
                        jQuery(document).trigger('heartbeat-tick', [{'wp-auth-check': false}]);
                    }
                },
                settings: <?php echo wp_json_encode($settings, JSON_PRETTY_PRINT); ?>,
                cmsSettings: <?php echo wp_json_encode($cms_settings, JSON_PRETTY_PRINT); ?>
            };

            jQuery(document).on("heartbeat-tick.wp-refresh-nonces", function(c, d) {
                if (d['wp-refresh-post-nonces'] && typeof dataBridge === 'object') {
                    dataBridge.settings.ajaxData._ajax_nonce = d['wp-refresh-post-nonces'].replace.np_ajax_nonce;
                    dataBridge.settings.uploadImageOptions.params._ajax_nonce = d['wp-refresh-post-nonces'].replace.np_ajax_nonce;
                    dataBridge.settings.uploadImageOptions.params._wpnonce = d['wp-refresh-post-nonces'].replace.np_upload_image_nonce;

                    attemptsCount++;
                    callbacks.forEach(function (cb) {
                        cb(true);
                    });
                    callbacks = [];
                }
            });
            jQuery(document).on('click', '#wp-auth-check-wrap .wp-auth-check-close', function() {
                attemptsCount++;
                callbacks.forEach(function (cb) {
                    cb(false);
                });
            });



        </script>
        <?php
    }

    /**
     * Create unique page title based on specified string
     *
     * @param string $title
     *
     * @return string
     */
    private static function _createUniqueTitle($title) {
        while (($p = get_page_by_title($title)) && $p->post_title === $title) {
            if (preg_match('#(.*\s)(\d+)$#', $title, $match)) {
                $new_title = $match[1] . ($match[2] + 1);
                if ($title === $new_title) {
                    break;
                }
                $title = $new_title;
            } else {
                $title = $title . ' 1';
            }
        }
        return $title;
    }

    /**
     * Action on in_admin_header
     * Prepare theme typography scripts
     */
    public static function printLoaderFrameScripts() {
        global $post;
        if (!$post || !NpEditor::isAllowedForEditor($post)) {
            return;
        }

        $defined_settings = apply_filters('np_theme_settings', array());
        if ($defined_settings) {
            ?>
            <script>
                dataBridgeData.info.themeTypography = <?php echo wp_json_encode($defined_settings['typography']); ?>;
                dataBridgeData.info.themeFontScheme = <?php echo wp_json_encode($defined_settings['fontScheme']); ?>;
                dataBridgeData.info.themeColorScheme = <?php echo wp_json_encode($defined_settings['colorScheme']); ?>;
            </script>
            <?php
            return;
        }
        ?>
        <script src="<?php echo APP_PLUGIN_URL; ?>/editor/assets/js/typography-parser.js?v=<?php echo APP_PLUGIN_VERSION; ?>"></script>
        <iframe id="np-loader"></iframe>

        <script>
            var loadCallback;

            var needResetCache = !localStorage.np_theme_typography_cache ||
                document.cookie.indexOf('np_theme_typography_cache_force_refresh=1') !== -1;

            if (needResetCache) {
                delete localStorage.np_theme_typography_cache;
            }

            window.loadAppHook = function (load) {
                if (localStorage.np_theme_typography_cache) {
                    jQuery.extend(dataBridgeData.info, JSON.parse(localStorage.np_theme_typography_cache));
                    console.log('Regular load app.js');
                    load();
                    return;
                }
                loadCallback = load;
            };

            var loaderIframe = document.getElementById('np-loader');
            loaderIframe.addEventListener("load", function() {
                localStorage.np_theme_typography_cache = JSON.stringify(NpTypographyParser.parse(loaderIframe));
                document.cookie = 'np_theme_typography_cache_force_refresh=';
                jQuery(loaderIframe).remove();
                console.log('Typography cache updated');
                jQuery.extend(dataBridgeData.info, JSON.parse(localStorage.np_theme_typography_cache));
                if (loadCallback) {
                    console.log('Deferred load app.js');
                    loadCallback();
                }
            });

            if (location.protocol === "https:" && dataBridgeData.info.typographyPageHtmlUrl.indexOf('http://') !== -1) {
                console.log('Regular load app.js due to CORS');
                delete window.loadAppHook;
            } else {
                loaderIframe.src = dataBridgeData.info.typographyPageHtmlUrl;
            }
        </script>
        <?php
    }

    /**
     * Filter on the_posts
     * Create fake post for testing typography
     *
     * @param array $posts
     *
     * @return stdClass[]
     */
    public static function fakePostsFilter($posts) {
        global $wp_query;

        $post = new stdClass;
        $post->ID = -1;
        $post->post_author = 1;
        $post->post_date = current_time('mysql');
        $post->post_date_gmt = current_time('mysql', 1);
        $post->post_content = '<div id="np-test-container"></div>';
        $post->post_title = 'TEST';
        $post->post_excerpt = '';
        $post->post_status = 'publish';
        $post->comment_status = 'closed';
        $post->ping_status = 'closed';
        $post->post_password = '';
        $post->post_name = 'test-theme-page';
        $post->to_ping = '';
        $post->pinged = '';
        $post->modified = $post->post_date;
        $post->modified_gmt = $post->post_date_gmt;
        $post->post_content_filtered = '';
        $post->post_parent = 0;
        $post->guid = get_home_url('/' . $post->post_name);
        $post->menu_order = 0;
        $post->post_tyle = 'page';
        $post->post_mime_type = '';
        $post->comment_count = 0;

        $wp_query->is_page = true;
        $wp_query->is_singular = true;
        $wp_query->is_home = false;
        $wp_query->is_archive = false;
        $wp_query->is_category = false;
        unset($wp_query->query['error']);
        $wp_query->query_vars['error'] = '';
        $wp_query->is_404 = false;

        return array($post);
    }


    /**
     * Filter on admin_url
     * Add domain parameter if need
     *
     * @param string $url
     *
     * @return string
     */
    public static function adminUrlFilter($url) {
        if (isset($_GET['domain'])) {
            $url = add_query_arg(array('domain' => urlencode($_GET['domain'])), $url);
        }
        return $url;
    }

    /**
     * Get fonts info
     *
     * @return array
     */
    public static function getFontsInfo() {
        $info = array(
            'path' => '',
            'canSave' => true,
        );
        $assets = APP_PLUGIN_PATH . 'assets/css/';
        if (file_exists($assets)) {
            $error = self::checkWritable($assets);
            if (count($error) > 0) {
                return array_merge($info, $error);
            }
            $fonts = $assets . '/fonts';
            if (!file_exists($fonts)) {
                if (false === @mkdir($fonts, 0777, true)) {
                    return array_merge($info, array('path' => $fonts, 'canSave' => false));
                }
            } else {
                $error = self::checkWritable($fonts);
                if (count($error) > 0) {
                    return array_merge($info, $error);
                }
            }
        }
        return $info;
    }

    /**
     * Check path writable
     *
     * @param string $path Path
     *
     * @return string
     */
    public static function checkWritable($path) {
        $user = get_current_user();
        @chown($path, $user);
        @chmod($path, 0777);
        $result = array();
        if (!is_writable($path)) {
            $result = array(
                'path' => $path,
                'canSave' => false,
            );
        }
        return $result;
    }

    /**
     * Get downloaded fonts
     *
     * @return false|string
     */
    public static function getDownloadedFonts() {
        $downloadedFontsFile = APP_PLUGIN_PATH . 'assets/css/fonts/downloadedFonts.json';
        return file_exists($downloadedFontsFile) ? file_get_contents($downloadedFontsFile) : '';
    }

    /**
     * Action on in_admin_header
     * Print Mediaspot_zopyron-editor iframe at the top
     */
    public static function printEditorAction() {
        global $post;
        if (_arr($_GET, 'np_new') || _arr($_GET, 'np_edit')) {
            if ($post && $post->ID && !_arr($_GET, 'np_new')) {
                np_data_provider($post->ID, true)->clear();
            }
            $can_start_editor = get_option('npDataBridge', 1);
            if ($can_start_editor) {
                if (_arr($_GET, 'np_page')) {
                    $settingsPage = array(
                        'post_id' => $post ? $post->ID : 0,
                        'page' => _arr($_GET, 'np_page'),
                    );
                } else {
                    $settingsPage = array('post_id' => $post ? $post->ID : 0);
                }
                echo NpAdminActions::getEditorContainerHtml($settingsPage);
            } else {
                ?>
                <script>
                    window.location.href = `<?php echo admin_url().'edit.php?post_type=page';?>`;
                    alert('Unable to start the Editor. Please contact the Support.');
                </script>
                <?php
            }
        }
    }

    /**
     * Action in init
     */
    public static function init() {
        add_filter('page_row_actions', 'NpAdminActions::pageRowAction');
    }

    /**
     * Action on page_row_actions
     * Add "Edit with Mediaspot-Zopyron" links to pages list
     *
     * @param array $actions
     *
     * @return array
     */
    public static function pageRowAction($actions) {
        $post = get_post();
        if (np_data_provider($post->ID)->isConvertRequired()) {
            $actions['edit_in_mediaspot_zopyron'] = '<a href="' . add_query_arg(array('np_edit' => '1'), get_edit_post_link($post->ID)) . '">' . __('Turn to Mediaspot-Zopyron', 'mediaspot_zopyron') . '</a>';
        } else {
            $actions['edit_in_mediaspot_zopyron'] = '<a href="' . add_query_arg(array('np_edit' => '1'), get_edit_post_link($post->ID)) . '">' . __('Edit with Mediaspot-Zopyron', 'mediaspot_zopyron') . '</a>';
        }
        return $actions;
    }

    /**
     * Action on wp_refresh_nonces
     * Add new nonces to replace them in mediaspot_zopyronSettings
     *
     * @param array  $response
     * @param array  $data
     * @param string $screen_id
     *
     * @return array
     */
    public static function refreshNonsesFilter($response, $data, $screen_id) {
        if (array_key_exists('wp-refresh-post-nonces', $data)) {
            if (!$post_id = absint($data['wp-refresh-post-nonces']['post_id'])) {
                return $response;
            }

            if (!current_user_can('edit_post', $post_id)) {
                return $response;
            }

            $response['wp-refresh-post-nonces']['replace']['np_ajax_nonce'] = wp_create_nonce('np-upload');
            $response['wp-refresh-post-nonces']['replace']['np_upload_image_nonce'] = wp_create_nonce('media-form');
        }

        return $response;
    }

    /**
     * Action on after_switch_theme
     * Set cookie for refreshing typography cache
     */
    public static function forceRefreshTypographyCache() {
        setcookie('np_theme_typography_cache_force_refresh', '1', time() + 3*YEAR_IN_SECONDS);
    }

    /**
     * @param array   $post_states
     * @param WP_Post $post
     * @return array  $post_states
     */
    public static function addPostState($post_states, $post) {
        $data_provider = np_data_provider($post->ID);
        if ($post->post_type === "page" && $data_provider->isMediaspot_zopyron()) {
            $post_states[] = 'Mediaspot_zopyron';
        }
        return $post_states;
    }

    /**
     * Restore page type for editor
     *
     * @param string  $pageHtml Page html
     * @param WP_Post $post
     *
     * @return mixed
     */
    private static function _restorePageType($pageHtml, $post) {
        if ($post->post_type === 'page') {
            $post_type = get_post_meta($post->ID, '_np_template', true);
            $pageView = isset($post_type) ? $post_type : NpSettings::getOption('np_template');
            $rePageType = '/<meta name="page_type" content="[^"]+?">/';
            if (preg_match($rePageType, $pageHtml)) {
                $pageHtml = preg_replace($rePageType, '<meta name="page_type" content="' . self::$_editorPageTypes[$pageView] . '">', $pageHtml);
            } else {
                $pageHtml = str_replace('<head>', '<head><meta name="page_type" content="' . self::$_editorPageTypes[$pageView] . '">', $pageHtml);
            }
        }
        return $pageHtml;
    }
}

// @codingStandardsIgnoreStart
/**
 * Action on edit_form_after_title
 * NOTE: For backward compatibility with themler-core do not rename this function
 *
 * @param WP_Post $post
 */
function upage_edit_form_buttons($post) {
    ob_start();
    do_action('themler_edit_form_buttons', $post);
    $html = ob_get_clean();

    if ($html) {
        ?>
        <div style="margin-top: 5px; margin-bottom: 10px;">
            <?php echo $html; ?>
        </div>
        <?php
    }
}
// @codingStandardsIgnoreEnd
if (!has_action('edit_form_after_title', 'themler_edit_form_buttons')) {
    add_action('edit_form_after_title', 'upage_edit_form_buttons');
}

add_filter('display_post_states', 'NpAdminActions::addPostState', 10, 2);
add_filter('get_edit_post_link', 'NpAdminActions::editPostLinkFilter');
add_action('edit_form_top', 'NpAdminActions::editFormTopFilter');
add_action('themler_edit_form_buttons', 'NpAdminActions::addMediaspot_zopyronButtonAction');
add_action('admin_menu', 'NpAdminActions::addEditorPageAction');
add_action('_network_admin_menu', 'NpAdminActions::addMenuAction');
add_action('_user_admin_menu', 'NpAdminActions::addMenuAction');
add_action('_admin_menu', 'NpAdminActions::addMenuAction');
add_action('admin_print_styles', 'NpAdminActions::printMenuStylesAction');
add_action('load-pages_page_np_app', 'NpAdminActions::appAction');
add_action('edit_form_after_title', 'NpAdminActions::showScreenShotsAction', 100);
add_action('admin_head', 'NpAdminActions::printPreviewStylesAction');
add_action('admin_head', 'NpAdminActions::printBridgeAction');
add_action('custom_menu_order', 'NpAdminActions::addSubmenusAction');
add_action('in_admin_header', 'NpAdminActions::printEditorAction');
add_action("in_admin_header", 'NpAdminActions::printLoaderFrameScripts');
add_action('init', 'NpAdminActions::init');

// after wp_refresh_post_nonces
add_filter('wp_refresh_nonces', 'NpAdminActions::refreshNonsesFilter', 11, 3);

add_action("after_switch_theme", 'NpAdminActions::forceRefreshTypographyCache');

if (Mediaspot_zopyron::isHtmlQuery()) {
    add_filter('the_posts', 'NpAdminActions::fakePostsFilter', 1003);
}